﻿using System;
using System.Drawing;
using System.Numerics;

namespace RayTracingTest
{
	enum DitherType
	{
		None,					//減色なし（フルカラー）
		OrderedDither,			//ディザマトリクス
		ErrorDiffusionDither	//誤差拡散
	}

	interface IDither
	{
		Color Dithering(int canvasX, int canvasY, Color color);
	}

	partial class RayTracing
	{
		private	struct Sphere
		{
			public Vector3 origin { get; private set; }
			public float radius { get; private set; }
			public Color color { get; set; }

			public Sphere(float x, float y, float z, float r, Color color)
			{
				origin = new Vector3(x, y, z);
				radius = r;
				this.color = color;
			}
		}

		private struct Screen
		{
			public int Width { get; private set; }
			public int Height { get; private set; }
			public float Z { get; private set; }
			public int Left { get; private set; }
			public int Right { get; private set; }
			public int Top { get; private set; }
			public int Bottom { get; private set; }

			public Screen(int width, int height, float z)
			{
				Width = width;
				Height = height;
				Z = z;

				//スクリーン座標
				Left = -width / 2;
				Right = Left + width - 1;
				Bottom = -height / 2;
				Top = Bottom + height - 1;
			}

			//スクリーン座標から描画座標へ変換する
			public Point ToCanvas(int screenX, int screenY)
			{
				int x = screenX - Left;     //-64..+63 -> 0..127
				int y = -(screenY - Top);   //+63..-64 -> 0..127
				return new Point(x, y);
			}
		}

		private struct VectorParam
		{
			public Vector3 S;   //視線ベクトル
			public Vector3 Px;  //視線と物体の交点座標
			public Vector3 Nx;  //交点に立つ法線ベクトル
			public Vector3 Lx;  //交点から光源へのベクトル
		}
	}

	partial class RayTracing
	{
		/*	座標軸の向き		:      +y  +z
			　左:-x ... +x:右	:　    ｜／
			　下:-y ... +y:上	: -x －＋－ +x
			手前:-z ... +z:奥	:    ／｜
								:  -z  -y
		*/
		/*	64x64
		private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 100.0f, Color.Orange);
		//private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 100.0f, Color.Gray);
		private readonly Screen screen = new Screen(64, 64, -200.0f);
		private readonly Vector3 view = new Vector3(0.0f, 0.0f, -280.0f);
		private readonly Sphere light = new Sphere(300.0f, 300.0f, -500.0f, 0.0f, Color.White);
		//*/
		/*	128x128
		private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 150.0f, Color.Orange);
		//private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 150.0f, Color.Gray);
		private readonly Screen screen = new Screen(128, 128, -200.0f);
		private readonly Vector3 view = new Vector3(0.0f, 0.0f, -300.0f);
		private readonly Sphere light = new Sphere(300.0f, 300.0f, -500.0f, 0.0f, Color.White);
		//*/
		//*	320x320
		private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 300.0f, Color.Orange);
		//private Sphere ball = new Sphere(0.0f, 0.0f, 0.0f, 300.0f, Color.Gray);
		private readonly Screen screen = new Screen(320, 320, -500.0f);
		private readonly Vector3 view = new Vector3(0.0f, 0.0f, -800.0f);
		private readonly Sphere light = new Sphere(300.0f, 300.0f, -800.0f, 0.0f, Color.White);
		//*/

		private readonly Color BackgroundColor = Color.Blue;
		private VectorParam vParam;

		private Action<Point, Color> dlgtDrawFunction;
		private IDither dither;

		//コンストラクタ
		public RayTracing(Action<Point, Color> drawFunction)
		{
			dlgtDrawFunction = drawFunction;
		}

		public Size GetScreenSize()
		{
			return new Size(screen.Width, screen.Height);
		}

		public void	Rendering(DitherType ditherType, bool isGrayScale)
		{
			switch (ditherType)
			{
			case DitherType.OrderedDither: dither = new OrderedDither(); break;
			case DitherType.ErrorDiffusionDither: dither = new ErrorDiffusionDither(screen.Width); break;
			default: dither = null; break;
			}

			ball.color = isGrayScale ? Color.Gray : Color.Orange;

			Color color;
			for (int screenY = screen.Top; screen.Bottom <= screenY; screenY--)    //+63..-64
			{
				for (int screenX = screen.Left; screenX <= screen.Right; screenX++)  //-64..+63
				{
					Point canvasPt = screen.ToCanvas(screenX, screenY);

					//視線と物体との交点計算
					if (Intersection(screenX, screenY))
					{
						//点の明るさ計算
						float dif = DiffuseReflection();
						float spec = SpecularReflection();
						float ambi = 0.2f;

						//点の色決定
						color = ColorDiffuse(ball.color, dif);
						color = ColorSpecular(color, light.color, spec);
						color = ColorAmbient(color, ball.color, ambi);

						//減色
						if (dither != null)
						{
							color = dither.Dithering(canvasPt.X, canvasPt.Y, color);
						}
					}
					else
					{
						color = BackgroundColor;
					}

					//点を描画する
					DrawPixel(canvasPt, color);
				}

				//誤差拡散ディザの場合、誤差累積バッファを入れ替える
				(dither as ErrorDiffusionDither)?.SwapBufferLine();
			}
		}

		//交点計算
		private bool Intersection(int screenX, int screenY)
		{
			/*
			V:視点（位置ベクトル）, S:視線（単位ベクトル）
			C:球体の中心（位置ベクトル）, r:球体の半径（スカラー）
			Px:交点（位置ベクトル）
			
			交点：	Px = V + S*t (t:媒介変数)	…(1)
			球面：	(Px - C)^2 = r^2			…(2)
			
			(1)を(2)へ代入し、tについて整理すると、
			(S^2)*t^2 + 2(V-C)・St + ((V-C)^2-r^2) = 0	…(3)交点を求める式
			-> (a)t^2 + 2(b)t + (c) = 0 の形

			(3)の判別式: D = b^2-a*c = ((V-C)・S)^2 - (S^2)*((V-C)^2-r^2)
			(3)の解: t = (-b +- √D)/a = -((V-C)・S) +- √D / (S^2)	…'+-'のうち、隠面消去を考慮すると必要なのは'-'の方の解（小さい値の方）
			*/

			//視線（視点からスクリーンへ向かうベクトル）
			Vector3 S = Vector3.Subtract(new Vector3((float)screenX, (float)screenY, screen.Z), view);
			S = Vector3.Normalize(S);

			//交差判定
			Vector3 vecCV = Vector3.Subtract(view, ball.origin);    //(V-C)はCからVへ向かうベクトルを意味する
			float a = InnerProduct(S, S);
			float b = InnerProduct(vecCV, S);
			float c = InnerProduct(vecCV, vecCV) - (ball.radius * ball.radius);
			float D = b * b - a * c;
			if (D < 0) { return false; }

			//交点の座標
			float t = (float)(-b - Math.Sqrt(D)) / a;
			Vector3 St = Vector3.Multiply(S, t);
			Vector3 P = Vector3.Add(view, St);

			//交点の法線ベクトル
			Vector3 N = Vector3.Subtract(P, ball.origin);
			N = Vector3.Normalize(N);

			//入射光のベクトル（始点は光源）
			Vector3 L = Vector3.Subtract(P, light.origin);
			L = Vector3.Normalize(L);

			vParam.S = S;
			vParam.Px = P;
			vParam.Nx = N;
			vParam.Lx = Vector3.Negate(L);  //向きを反転し、交点を始点とする

			return true;
		}

		//拡散反射
		private float DiffuseReflection()
		{
			/*
			物体表面に陰影を付ける（紙のような質感表現）
			視線と物体の交点に法線を立てる。
			交点の明るさは入射光と法線とのなす角Aの余弦(cosA)に比例する。
			→例として、北極点に法線が立ち、その先に光源がある場合、
			　北極はcosA=1.0、赤道上はcosA=0.0、南半球はcosA<0.0（光が当たらない）。
			ここでは簡略的に「cosAの値 ＝ 物体の地色を何％発色させるか」と考えることにする。
			
			内積：　A・B = |A||B|cosX			…(1)
			ベクトルの大きさ　|A| = √(A・A)	…(2)
			
			(2)を(1)へ代入して整理すると	cosX = (A・B) / (√(A・A) * √(B・B))
			*/

			float AB = InnerProduct(vParam.Lx, vParam.Nx);
			float AA = InnerProduct(vParam.Lx, vParam.Lx);
			float BB = InnerProduct(vParam.Nx, vParam.Nx);
			float cosX = (float)(AB / (Math.Sqrt(AA * BB)));
			if (cosX < 0.0f) { cosX = 0.0f; }

			return cosX;
		}

		//鏡面反射
		private float SpecularReflection()
		{
			/*
			物体表面にハイライトを付ける（ツルピカな質感表現）
			ハイライトの明るさは反射光と視線のなす角Aの余弦(cosA)に依存する。
			ハイライトの広がりはcosAをn乗して調整する（硬質感の表現）。
			ここでは簡略的に「cosAのn乗 ＝ 光源の色が何％反映されるか」と考えることにする。
			
			N:交点の法線, L:入射光(ただし交点が始点), R:反射光, B:入射光と法線がなす角
			とすると、これらの関係は、
			L + R = 2(N*cosB)	…(1)
			L・N = |L||N|cosB	…(2)
			(1)(2)よりcosBを消去すると	R = 2N((L・N) / (|L||N|)) - L	…(3)
			
			視線(ただし交点が始点)をSとすると	cosA = (R・S) / (|R||S|)	…(4)
			(3)を(4)へ代入することで既知のベクトルN,L,SからcosAが求まる。
			*/

			float LN = InnerProduct(vParam.Lx, vParam.Nx);
			float LL = InnerProduct(vParam.Lx, vParam.Lx);
			float NN = InnerProduct(vParam.Nx, vParam.Nx);
			float cosB = (float)(LN / (Math.Sqrt(LL * NN)));
			if (cosB < 0.0f) { return 0.0f; }   //入射光が交点に当たっていない

			Vector3 M = Vector3.Multiply(2 * cosB, vParam.Nx); // 2(N*cosB)
			Vector3 R = Vector3.Subtract(M, vParam.Lx);
			Vector3 Sx = Vector3.Negate(vParam.S);   //視線ベクトルは交点を始点とする向きで使用する

			float RS = InnerProduct(R, Sx);
			float RR = InnerProduct(R, R);
			float SS = InnerProduct(Sx, Sx);
			float cosA = (float)(RS / (Math.Sqrt(RR * SS)));
			if (cosA < 0.0f) { cosA = 0.0f; }   //この視線ではハイライトは見えない

			cosA = (float)Math.Pow(cosA, 20);   // (cosA)^n
			return cosA;
		}

		//拡散反射を反映させた色を作る（地色を何％の強さで発色させるか）
		//0.0 <= rate <= 1.0
		private Color ColorDiffuse(Color baseColor, float difRate)
		{
			byte rr = (byte)(baseColor.R * difRate);
			byte gg = (byte)(baseColor.G * difRate);
			byte bb = (byte)(baseColor.B * difRate);
			return Color.FromArgb(rr, gg, bb);
		}

		//鏡面反射を反映させた色を作る（光源の色が何％占めるか）
		//0.0 <= rate <= 1.0
		private Color ColorSpecular(Color currentColor, Color light, float specRate)
		{
			byte rr = (byte)Math.Min(0xFF, light.R * specRate + currentColor.R * (1.0f - specRate));
			byte gg = (byte)Math.Min(0xFF, light.G * specRate + currentColor.G * (1.0f - specRate));
			byte bb = (byte)Math.Min(0xFF, light.B * specRate + currentColor.B * (1.0f - specRate));
			return Color.FromArgb(rr, gg, bb);
		}

		//環境光により発色を底上げする（地色の何％の濃さを最低限発色させるか）
		//0.0 <= rate <= 1.0
		private Color ColorAmbient(Color currentColor, Color baseColor, float ambiRate)
		{
			byte rr = (byte)Math.Min(0xFF, currentColor.R + baseColor.R * ambiRate);
			byte gg = (byte)Math.Min(0xFF, currentColor.G + baseColor.G * ambiRate);
			byte bb = (byte)Math.Min(0xFF, currentColor.B + baseColor.B * ambiRate);
			return Color.FromArgb(rr, gg, bb);
		}

		//内積
		private float InnerProduct(Vector3 a, Vector3 b)
		{
			//a・b = aXbX + aYbY + aZbZ
			return Vector3.Dot(a, b);
		}

		//点を描画する
		private void DrawPixel(Point canvasPt, Color color)
		{
			dlgtDrawFunction(canvasPt, color);
		}
	}
}
