﻿using System.Drawing;

namespace RayTracingTest
{
	class ErrorDiffusionDither : IDither
	{
		/*	【Floyd-Steinberg dithering】
			| |*|A|		CurrentLine
			|B|C|D|		NextLine
			
			誤差拡散の重み付け：A=7/16, B=3/16, C=5/16, D=1/16
		*/
		private readonly int[] DiffusionWait = { 7, 3, 5, 1 }; // n/16
		private const int CR = 0;	//Current_line:Right 
		private const int NL = 1, ND = 2, NR = 3;   //Next_line:Left,Down,Right

		//誤差累積配列
		private int[][][] errorBuffer = new int[2][][];
		private const int CurrentLine = 0, NextLine = 1;
		private const int BufR = 0, BufG = 1, BufB = 2;

		private readonly int ScreenWidth;

		//コンストラクタ
		public ErrorDiffusionDither(int screenWidth)
		{
			ScreenWidth = screenWidth;
			errorBuffer[CurrentLine] = CreateBufferLine();
			errorBuffer[NextLine] = CreateBufferLine();
		}

		private int[][] CreateBufferLine()
		{
			int[][] buf = new int[ScreenWidth + 1][];	//右への誤差の配分処理を簡単にするため'+1'

			for (int x = 0; x < buf.Length; x++)
			{
				buf[x] = new int[] { 0, 0, 0 };
			}

			return buf;
		}

		public void SwapBufferLine()
		{
			errorBuffer[CurrentLine] = errorBuffer[NextLine];
			errorBuffer[NextLine] = CreateBufferLine();		//ゼロクリアが面倒なので新たに生成する
		}

		public Color Dithering(int canvasX, int canvasY, Color color)
		{
			byte rr = Distribute2(color.R, canvasX, BufR);
			byte gg = Distribute2(color.G, canvasX, BufG);
			byte bb = Distribute2(color.B, canvasX, BufB);
			return Color.FromArgb(rr, gg, bb);
		}

		//色の振り分け（2階調）
		private byte Distribute2(byte colorElement, int x, int idxRgb)
		{
			int currentColor = colorElement + errorBuffer[CurrentLine][x][idxRgb];

			int threshold = 0x80;
			int newColor = (threshold <= currentColor) ? 0xFF : 0x00;
			int quantizationError = currentColor - newColor;

			//右への分配
			errorBuffer[CurrentLine][x + 1][idxRgb] += quantizationError * DiffusionWait[CR] / 16;

			//左下への分配
			if (1 <= x)
			{
				errorBuffer[NextLine][x - 1][idxRgb] += quantizationError * DiffusionWait[NL] / 16;
			}

			//真下への分配
			errorBuffer[NextLine][x][idxRgb] += quantizationError * DiffusionWait[ND] / 16;

			//右下への分配
			errorBuffer[NextLine][x + 1][idxRgb] += quantizationError * DiffusionWait[NR] / 16;

			return (byte)newColor;
		}
	}
}
